﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq.Expressions;
using DA = System.ComponentModel.DataAnnotations;

namespace gov.va.med.vbecs.Common
{
    /// <summary>
    /// Used to notifiy property changes and data validation errors
    /// </summary>
    public class NotifyPropertyChanged : INotifyPropertyChanged, IDataErrorInfo
    {
        private Dictionary<string, ICollection<DA.ValidationResult>> _propertyErrors { get; set; }
        /// <summary>
        /// Event that fires property changes
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// NotifyPropertyChanged constructor
        /// </summary>
        public NotifyPropertyChanged()
        {
            _propertyErrors = new Dictionary<string, ICollection<DA.ValidationResult>>();
        }

        /// <summary>
        /// RaisePropertyChanged
        /// </summary>
        /// <param name="propertyName"></param>
        public void RaisePropertyChanged(string propertyName)
        {
            if (this.PropertyChanged != null)
            {
                this.PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// RaisePropertyChanged with lambda expression
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="propertyExpression"></param>
        public void RaisePropertyChanged<T>(Expression<Func<T>> propertyExpression)
        {
            if (this.PropertyChanged != null)
            {
                this.PropertyChanged(this, new PropertyChangedEventArgs(NameOf(propertyExpression)));
            }
        }

        /// <summary>
        /// Validate property and raise property changed
        /// </summary>
        /// <param name="propertyName"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public string ValidateAndRaisePropertyChanged(string propertyName, object value)
        {
            var returnString = ValidateProperty(propertyName, value);
            RaisePropertyChanged(propertyName);
            return returnString;
        }

        /// <summary>
        /// Validate property and raise property changed with lambda expression for property name
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="expression"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public string ValidateAndRaisePropertyChanged<T>(Expression<Func<T>> expression, object value)
        {
            var returnString = ValidateProperty(expression, value);
            RaisePropertyChanged(expression);
            return returnString;
        }

        /// <summary>
        /// Validate property against data annotation validation rule
        /// </summary>
        /// <param name="propertyName"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public virtual string ValidateProperty(string propertyName, object value)
        {
            ValidationContext context = new ValidationContext(this, null, null);
            context.MemberName = propertyName;
            ICollection<DA.ValidationResult> errorCollection;

            lock (_propertyErrors)
            {
                // Add the key if it doens't already exist
                if (!_propertyErrors.ContainsKey(propertyName))
                {
                    _propertyErrors.Add(propertyName, new Collection<DA.ValidationResult>());
                }

                errorCollection = _propertyErrors[propertyName];

                if (errorCollection != null)
                {
                    // always start fresh with no errors
                    errorCollection.Clear();

                    // Validate the property and fill the errorCollection
                    Validator.TryValidateProperty(value, context, errorCollection);
                }
            }

            // Return the error message(s) if any exist
            return GetErrorsFromCollection(errorCollection);
        }

        /// <summary>
        /// Validate property against data annotation validation rule
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="propertyExpression"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        protected string ValidateProperty<T>(Expression<Func<T>> propertyExpression, object value)
        {
            return ValidateProperty(NameOf(propertyExpression), value);
        }

        #region IDataErrorInfo
        /// <summary>
        /// Check if errors exist in this class's instance
        /// </summary>
        public bool HasErrors
        {
            get
            {
                return PropertyErrorsExist();
            }
        }

        private bool PropertyErrorsExist()
        {
            bool errors = false;

            foreach (var propertyName in _propertyErrors.Keys)
            {
                ICollection<DA.ValidationResult> errorCollection = GetErrorsCollection(propertyName);

                if (errorCollection != null
                    && errorCollection.Count > 0)
                {
                    errors = true;
                    break;
                }
            }

            return errors;
        }

        /// <summary>
        /// Error
        /// </summary>
        public string Error
        {
            get { return null; }
        }

        /// <summary>
        /// Check property for errors
        /// </summary>
        /// <param name="columnName"></param>
        /// <returns></returns>
        public string this[string columnName]
        {
            get
            {
                return GetErrors(columnName);
            }
        }

        /// <summary>
        /// Get errors for given property name
        /// </summary>
        /// <param name="propertyName"></param>
        /// <returns></returns>
        public string GetErrors(string propertyName)
        {
            string errors = null;

            lock (_propertyErrors)
            {
                ICollection<DA.ValidationResult> errorCollection = GetErrorsCollection(propertyName);

                if (errorCollection != null)
                {
                    errors = GetErrorsFromCollection(errorCollection);
                }
            }

            // Not sure RaisePropertyChanged is really needed since a DelegateCommand (e.g. Save Button)
            // should be setting it's CanExecute method to PropertyErrorsExist()
            // RaisePropertyChanged("HasErrors");
            return errors;
        }

        /// <summary>
        /// Get Errors in a collection
        /// </summary>
        /// <param name="propertyName"></param>
        /// <returns></returns>
        public ICollection<DA.ValidationResult> GetErrorsCollection(string propertyName)
        {
            lock (_propertyErrors)
            {
                if (_propertyErrors.ContainsKey(propertyName))
                {
                    return _propertyErrors[propertyName];
                }
            }

            return null;
        }

        /// <summary>
        /// Get errors formatted from collection of errors
        /// </summary>
        /// <param name="errorCollection"></param>
        /// <returns></returns>
        private static string GetErrorsFromCollection(ICollection<DA.ValidationResult> errorCollection)
        {
            string errors = "";

            if (errorCollection != null)
            {
                foreach (DA.ValidationResult result in errorCollection)
                {
                    //var compositeResult = result as ValidationResult;
                    //if (compositeResult != null)
                    //{
                    //    foreach (var simpleResult in compositeResult.Results)
                    //    {
                    //        errors += simpleResult.ErrorMessage + "\n";
                    //    }
                    //}
                    //else
                    //{
                    errors += result.ErrorMessage + "\n";
                    //}
                }

                if (!string.IsNullOrWhiteSpace(errors))
                {
                    // Trim the last line break '\n'
                    errors = errors.TrimEnd('\n');
                }
            }

            return errors;
        }

        /// <summary>
        /// Remove all errors
        /// </summary>
        public void RemoveAllErrors()
        {

            lock (_propertyErrors)
            {
                IList<string> propertyNameList = new List<string>();

                foreach (KeyValuePair<string, ICollection<DA.ValidationResult>> keyValuePair in _propertyErrors)
                {
                    propertyNameList.Add(keyValuePair.Key);
                }

                _propertyErrors.Clear();

                foreach (var propertyName in propertyNameList)
                    RaisePropertyChanged(propertyName);
            }
        }

        /// <summary>
        /// Remove errors for the specified property 
        /// </summary>
        /// <param name="propertyName"></param>
        public void RemoveError(string propertyName)
        {
            lock (_propertyErrors)
            {
                if (_propertyErrors.ContainsKey(propertyName))
                    _propertyErrors.Remove(propertyName);
            }
            RaisePropertyChanged(propertyName);
        }

        /// <summary>
        /// Remove errors for the specified property 
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="propertyExpression"></param>
        public void RemoveError<T>(Expression<Func<T>> propertyExpression)
        {
            RemoveError(NameOf(propertyExpression));
        }

        /// <summary>
        /// Get string Name of property lambda expression
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="expression"></param>
        /// <returns></returns>
        public static string NameOf<T>(Expression<Func<T>> expression)
        {
            var memberExpr = expression.Body as MemberExpression;
            if (memberExpr == null) throw new ArgumentException("expression must be a Property or Member expression");
            return memberExpr.Member.Name;
        }

        #endregion
    }
}
